<?php
defined( 'ABSPATH' ) || exit;

class HRM_Payroll {

    public static function init() {
        add_action( 'wp_ajax_hrm_save_salary_structure', [ __CLASS__, 'handle_save_salary_structure' ] );
        add_action( 'wp_ajax_hrm_generate_payroll', [ __CLASS__, 'handle_generate_payroll' ] );
        add_action( 'wp_ajax_hrm_bulk_generate_payroll', [ __CLASS__, 'handle_bulk_generate_payroll' ] );
        add_action( 'wp_ajax_hrm_update_payroll_status', [ __CLASS__, 'handle_update_payroll_status' ] );
        add_action( 'wp_ajax_hrm_get_payslip', [ __CLASS__, 'handle_get_payslip' ] );
        add_action( 'wp_ajax_hrm_delete_payroll', [ __CLASS__, 'handle_delete_payroll' ] );
        add_action( 'wp_ajax_hrm_save_payroll_settings', [ __CLASS__, 'handle_save_payroll_settings' ] );
        add_action( 'wp_ajax_hrm_generate_demo_data', [ __CLASS__, 'handle_generate_demo_data' ] );
        add_action( 'wp_ajax_hrm_clear_demo_data', [ __CLASS__, 'handle_clear_demo_data' ] );
    }

    private static function render_payroll_settings() {
        $settings = get_option('hrm_payroll_settings', [
            'working_days' => 22,
            'tax_slabs' => [],
            'overtime_mode' => 'fixed',
            'currency' => '$'
        ]);
        ?>
        <div class="hrm-card-enhanced" style="max-width: 800px; margin: 0 auto;">
            <div class="hrm-card-header-enhanced">
                <h3>Payroll Configuration</h3>
                <p style="margin:4px 0 0; color:var(--hrm-text-secondary); font-size:13px;">Set global rules for salary calculation and taxes.</p>
            </div>
            <div class="hrm-card-body-enhanced">
                <form id="hrm-payroll-settings-form">
                    <?php wp_nonce_field('hrm_ajax_nonce', 'nonce'); ?>
                    <input type="hidden" name="action" value="hrm_save_payroll_settings">
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 24px; margin-bottom: 30px;">
                        <div class="hrm-form-group">
                            <label class="hrm-form-label">Standard Working Days (Month)</label>
                            <input type="number" name="working_days" class="hrm-input" value="<?php echo esc_attr($settings['working_days']); ?>" required>
                            <span class="hrm-form-hint">Used to calculate per-day salary for deductions.</span>
                        </div>
                        <div class="hrm-form-group">
                            <label class="hrm-form-label">Currency Symbol</label>
                            <input type="text" name="currency" class="hrm-input" value="<?php echo esc_attr($settings['currency']); ?>" required>
                            <span class="hrm-form-hint">Symbol shown across all payroll modules.</span>
                        </div>
                    </div>

                    <div style="background: var(--hrm-gray-50); padding: 24px; border-radius: var(--hrm-radius-lg); border: 1px solid var(--hrm-border-default);">
                        <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 20px;">
                            <div>
                                <h4 style="margin: 0; font-size: 16px; color: var(--hrm-text-primary);">Income Tax Slabs (Annual)</h4>
                                <p style="margin: 4px 0 0; font-size: 13px; color: var(--hrm-text-secondary);">Define tax percentage for different annual income ranges.</p>
                            </div>
                            <button type="button" class="hrm-payroll-btn hrm-payroll-btn-secondary btn-sm" id="add-tax-slab">
                                <span class="dashicons dashicons-plus"></span> Add Slab
                            </button>
                        </div>
                        
                        <div id="tax-slabs-list">
                            <?php if(!empty($settings['tax_slabs'])): foreach($settings['tax_slabs'] as $slab): ?>
                                <div class="slab-row" style="display:grid; grid-template-columns: 1fr 1fr 1fr 42px; gap:12px; margin-bottom:12px; align-items: center;">
                                    <input type="number" name="slab_min[]" class="hrm-input" placeholder="Min Income" value="<?php echo $slab['min']; ?>">
                                    <input type="number" name="slab_max[]" class="hrm-input" placeholder="Max Income" value="<?php echo $slab['max']; ?>">
                                    <input type="number" name="slab_percent[]" class="hrm-input" placeholder="Tax %" value="<?php echo $slab['percent']; ?>">
                                    <button type="button" class="remove-slab hrm-payroll-btn hrm-payroll-btn-danger btn-sm" style="padding: 10px; justify-content: center; height: 42px; background: #fee2e2; color: #ef4444; border-color: #fecaca;">&times;</button>
                                </div>
                            <?php endforeach; endif; ?>
                        </div>
                    </div>

                    <div style="margin-top: 30px; border-top: 1px solid var(--hrm-border-light); padding-top: 24px; display: flex; justify-content: flex-end;">
                        <button type="submit" class="hrm-payroll-btn hrm-payroll-btn-primary">
                            <span class="dashicons dashicons-saved"></span> Save Configuration
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <script>
        jQuery(document).ready(function($) {
            $('#add-tax-slab').on('click', function() {
                $('#tax-slabs-list').append(`
                    <div class="slab-row" style="display:flex; gap:10px; margin-bottom:10px;">
                        <input type="number" name="slab_min[]" placeholder="Min Income">
                        <input type="number" name="slab_max[]" placeholder="Max Income">
                        <input type="number" name="slab_percent[]" placeholder="Tax %">
                        <button type="button" class="remove-slab">&times;</button>
                    </div>
                `);
            });
            $(document).on('click', '.remove-slab', function() { $(this).parent().remove(); });

        });
        </script>
        <?php
    }

    public static function handle_save_payroll_settings() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $slabs = [];
        if (!empty($_POST['slab_min'])) {
            foreach ($_POST['slab_min'] as $i => $min) {
                $slabs[] = [
                    'min' => floatval($min),
                    'max' => floatval($_POST['slab_max'][$i]),
                    'percent' => floatval($_POST['slab_percent'][$i])
                ];
            }
        }

        $settings = [
            'working_days' => intval($_POST['working_days']),
            'currency' => sanitize_text_field($_POST['currency']),
            'tax_slabs' => $slabs
        ];

        update_option('hrm_payroll_settings', $settings);
        wp_send_json_success('Settings saved.');
    }

    public static function render_payroll() {
        $tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'dashboard';
        ?>
        <div class="wrap hrm-wrap hrm-payroll-module">
            <div class="hrm-header-section">
                <div class="hrm-header-left" style="display: flex; align-items: center; gap: 20px;">
                     <div style="width: 60px; height: 60px; background: var(--hrm-gradient-primary); border-radius: 12px; display: flex; align-items: center; justify-content: center; color: white; box-shadow: var(--hrm-shadow-md);">
                        <span class="dashicons dashicons-money-alt" style="font-size: 32px; width: 32px; height: 32px;"></span>
                     </div>
                     <div>
                        <h1 style="margin: 0; line-height: 1;">Payroll Management</h1>
                        <p style="margin: 8px 0 0;">Manage employee salaries, generate monthly payroll, and track payslips.</p>
                     </div>
                </div>
            </div>

            <nav class="hrm-nav-tabs">
                <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=dashboard'); ?>" class="hrm-nav-link <?php echo $tab === 'dashboard' ? 'active' : ''; ?>">Dashboard</a>
                <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=salary_structures'); ?>" class="hrm-nav-link <?php echo $tab === 'salary_structures' ? 'active' : ''; ?>">Salary Structures</a>
                <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=generate'); ?>" class="hrm-nav-link <?php echo $tab === 'generate' ? 'active' : ''; ?>">Generate Payroll</a>
                <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=history'); ?>" class="hrm-nav-link <?php echo $tab === 'history' ? 'active' : ''; ?>">Payroll History</a>
                <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=settings'); ?>" class="hrm-nav-link <?php echo $tab === 'settings' ? 'active' : ''; ?>">Settings</a>
            </nav>

            <div class="hrm-tab-content">
                <?php
                switch ( $tab ) {
                    case 'salary_structures':
                        self::render_salary_structures();
                        break;
                    case 'generate':
                        self::render_generate_payroll();
                        break;
                    case 'history':
                        self::render_payroll_history();
                        break;
                    case 'settings':
                        self::render_payroll_settings();
                        break;
                    case 'dashboard':
                    default:
                        self::render_dashboard();
                        break;
                }
                ?>
            </div>

            <!-- Global Payslip Modal -->
            <div id="hrm-payslip-modal" class="hrm-modal" style="display:none;">
                <div class="hrm-modal-content hrm-premium-modal hrm-payslip-container" style="max-width: 900px;">
                    <div class="hrm-modal-header-premium no-print">
                        <div class="header-icon-wrapper"><span class="dashicons dashicons-money-alt"></span></div>
                        <div class="header-text">
                            <h2>Employee Payslip</h2>
                            <p>Official Earnings & Deductions Statement</p>
                        </div>
                        <div class="header-actions" style="margin-left: auto; display: flex; gap: 10px;">
                            <button class="hrm-btn-outline btn-sm" onclick="window.print()"><span class="dashicons dashicons-printer"></span> Print Statement</button>
                            <button class="hrm-modal-close-premium" onclick="jQuery('#hrm-payslip-modal').fadeOut()"><span class="dashicons dashicons-no-alt"></span></button>
                        </div>
                    </div>
                    <div id="payslip-content" class="printable-area">
                        <!-- Loaded via AJAX -->
                    </div>
                </div>
            </div>

            <script>
            jQuery(document).ready(function($) {
                $(document).on('click', '.hrm-view-payslip', function() {
                    const id = $(this).data('id');
                    $('#payslip-content').html('<div style="text-align:center; padding:100px;">' + hrmLoadingSpinner() + '<p style="margin-top:15px; color:var(--text-secondary);">Generating payslip preview...</p></div>');
                    $('#hrm-payslip-modal').fadeIn().css('display', 'flex');
                    
                    $.post(ajaxurl, {
                        action: 'hrm_get_payslip',
                        id: id,
                        nonce: '<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>'
                    }, function(res) {
                        if (res.success) {
                            $('#payslip-content').html(res.data);
                        } else {
                            $('#payslip-content').html('<div class="hrm-alert hrm-alert-error">Failed to load payslip: ' + res.data + '</div>');
                        }
                    });
                });

                $(document).on('click', '.hrm-approve-payroll', function() {
                    if (!confirm('Approve this payroll? This will finalize the amount and make it visible to the employee.')) return;
                    const btn = $(this);
                    btn.prop('disabled', true).addClass('loading');
                    $.post(ajaxurl, {
                        action: 'hrm_update_payroll_status',
                        id: btn.data('id'),
                        status: 'approved',
                        nonce: '<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>'
                    }, function(res) {
                        if (res.success) {
                            hrmToast('success', 'Payroll approved successfully.');
                            setTimeout(() => location.reload(), 1000);
                        }
                    });
                });

                function hrmLoadingSpinner() {
                    return '<div class="hrm-loading-spinner" style="width: 40px; height: 40px; border-width: 3px; margin: 0 auto;"></div>';
                }
            });
            </script>
        </div>
        <?php
    }

    private static function render_dashboard() {
        global $wpdb;
        $month = date('Y-m');
        
        // Check for SQL errors and table existence
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}hrm_payroll'") === $wpdb->prefix . 'hrm_payroll';
        
        if (!$table_exists) {
            ?>
            <div class="hrm-card-enhanced" style="margin: 40px auto; max-width: 800px;">
                <div class="hrm-card-body-enhanced" style="text-align: center; padding: 60px 40px;">
                    <div style="font-size: 64px; color: #e74c3c; margin-bottom: 20px;">⚠️</div>
                    <h2 style="color: #e74c3c; margin-bottom: 15px;">Database Table Missing</h2>
                    <p style="color: var(--hrm-text-secondary); margin-bottom: 30px;">
                        The payroll database table does not exist. Please deactivate and reactivate the plugin to create the required tables.
                    </p>
                    <a href="<?php echo admin_url('plugins.php'); ?>" class="hrm-payroll-btn hrm-payroll-btn-primary">
                        <span class="dashicons dashicons-admin-plugins"></span> Go to Plugins
                    </a>
                </div>
            </div>
            <?php
            return;
        }
        
        // Comprehensive Payroll Calculations
        $stats = $wpdb->get_row( $wpdb->prepare( "
            SELECT 
                SUM(basic_salary + total_allowances + overtime_pay - attendance_penalty) as total_gross,
                SUM(net_salary) as total_net,
                SUM(CASE WHEN status = 'approved' THEN net_salary ELSE 0 END) as total_paid,
                SUM(CASE WHEN status = 'draft' THEN net_salary ELSE 0 END) as total_pending,
                COUNT(id) as processed_count
            FROM {$wpdb->prefix}hrm_payroll 
            WHERE payroll_month = %s", 
        $month ) );

        // Check for SQL errors
        if ($wpdb->last_error) {
            ?>
            <div class="hrm-card-enhanced" style="margin: 40px auto; max-width: 800px;">
                <div class="hrm-card-body-enhanced">
                    <div class="hrm-alert hrm-alert-error">
                        <strong>Database Error:</strong> <?php echo esc_html($wpdb->last_error); ?>
                    </div>
                </div>
            </div>
            <?php
            return;
        }

        $gross = $stats->total_gross ?: 0;
        $paid = $stats->total_paid ?: 0;
        $pending = $stats->total_pending ?: 0;
        $net = $stats->total_net ?: 0;
        $processed_count = $stats->processed_count ?: 0;
        
        // Check if there's any data
        $has_payroll_data = $processed_count > 0;
        
        // Get employee and salary structure counts for guidance
        $total_employees = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
        $employees_with_salary = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_salary_structures");

        ?>
        <div class="hrm-payroll-dashboard-enhanced">
            <!-- Stats Overview -->
            <div class="hrm-payroll-stats-row">
                <div class="hrm-payroll-stat-card-enhanced">
                    <div class="hrm-payroll-stat-icon blue">
                        <span class="dashicons dashicons-money-alt"></span>
                    </div>
                    <div class="hrm-payroll-stat-info">
                        <span class="hrm-payroll-stat-label">Total Gross Payroll</span>
                        <span class="hrm-payroll-stat-value"><?php echo hrm_currency_symbol() . number_format($gross, 2); ?></span>
                    </div>
                </div>
                
                <div class="hrm-payroll-stat-card-enhanced">
                    <div class="hrm-payroll-stat-icon green">
                        <span class="dashicons dashicons-yes-alt"></span>
                    </div>
                    <div class="hrm-payroll-stat-info">
                        <span class="hrm-payroll-stat-label">Approved & Paid</span>
                        <span class="hrm-payroll-stat-value"><?php echo hrm_currency_symbol() . number_format($paid, 2); ?></span>
                    </div>
                </div>

                <div class="hrm-payroll-stat-card-enhanced">
                    <div class="hrm-payroll-stat-icon yellow">
                        <span class="dashicons dashicons-clock"></span>
                    </div>
                    <div class="hrm-payroll-stat-info">
                        <span class="hrm-payroll-stat-label">Pending Approval</span>
                        <span class="hrm-payroll-stat-value"><?php echo hrm_currency_symbol() . number_format($pending, 2); ?></span>
                    </div>
                </div>

                <div class="hrm-payroll-stat-card-enhanced">
                    <div class="hrm-payroll-stat-icon purple">
                        <span class="dashicons dashicons-chart-area"></span>
                    </div>
                    <div class="hrm-payroll-stat-info">
                        <span class="hrm-payroll-stat-label">Total Net Salary</span>
                        <span class="hrm-payroll-stat-value"><?php echo hrm_currency_symbol() . number_format($net, 2); ?></span>
                    </div>
                </div>
            </div>

            <?php if (!$has_payroll_data): ?>
            <!-- Empty State Guidance -->
            <div class="hrm-card-enhanced" style="margin: 30px 0; border: 2px dashed var(--hrm-border-default); background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);">
                <div class="hrm-card-body-enhanced" style="text-align: center; padding: 50px 40px;">
                    <div style="font-size: 72px; margin-bottom: 20px; opacity: 0.3;">📊</div>
                    <h2 style="color: var(--hrm-text-primary); margin-bottom: 15px; font-size: 24px;">No Payroll Data for <?php echo date('F Y', strtotime($month . '-01')); ?></h2>
                    <p style="color: var(--hrm-text-secondary); margin-bottom: 30px; font-size: 15px; max-width: 600px; margin-left: auto; margin-right: auto;">
                        Get started by setting up salary structures for your employees and generating payroll records.
                    </p>
                    
                    <div style="background: white; border-radius: 8px; padding: 30px; margin: 30px auto; max-width: 700px; box-shadow: 0 2px 8px rgba(0,0,0,0.05); text-align: left;">
                        <h3 style="margin: 0 0 20px; color: var(--hrm-text-primary); font-size: 18px;">
                            <span class="dashicons dashicons-list-view" style="color: var(--hrm-primary);"></span> Quick Setup Guide
                        </h3>
                        
                        <div style="display: grid; gap: 20px;">
                            <?php if ($total_employees == 0): ?>
                            <div style="display: flex; gap: 15px; padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px;">
                                <div style="font-size: 24px; flex-shrink: 0;">⚠️</div>
                                <div>
                                    <strong style="color: #856404; display: block; margin-bottom: 5px;">No Employees Found</strong>
                                    <p style="margin: 0; color: #856404; font-size: 14px;">You need to add employees first before setting up payroll.</p>
                                    <a href="<?php echo admin_url('admin.php?page=hrm-employees'); ?>" class="hrm-payroll-btn hrm-payroll-btn-primary" style="margin-top: 10px; display: inline-flex;">
                                        <span class="dashicons dashicons-plus"></span> Add Employees
                                    </a>
                                </div>
                            </div>
                            <?php elseif ($employees_with_salary == 0): ?>
                            <div style="display: flex; gap: 15px; padding: 15px; background: #d1ecf1; border-left: 4px solid #17a2b8; border-radius: 4px;">
                                <div style="font-size: 24px; flex-shrink: 0;">1️⃣</div>
                                <div style="flex: 1;">
                                    <strong style="color: #0c5460; display: block; margin-bottom: 5px;">Step 1: Configure Salary Structures</strong>
                                    <p style="margin: 0 0 10px; color: #0c5460; font-size: 14px;">Define base salary, allowances, and deductions for each employee.</p>
                                    <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=salary_structures'); ?>" class="hrm-payroll-btn hrm-payroll-btn-primary">
                                        <span class="dashicons dashicons-admin-generic"></span> Set Up Salary Structures
                                    </a>
                                </div>
                            </div>
                            <?php else: ?>
                            <div style="display: flex; gap: 15px; padding: 15px; background: #d4edda; border-left: 4px solid #28a745; border-radius: 4px;">
                                <div style="font-size: 24px; flex-shrink: 0;">✓</div>
                                <div>
                                    <strong style="color: #155724; display: block; margin-bottom: 5px;">Salary Structures Configured</strong>
                                    <p style="margin: 0; color: #155724; font-size: 14px;"><?php echo $employees_with_salary; ?> out of <?php echo $total_employees; ?> employees have salary structures set up.</p>
                                </div>
                            </div>
                            
                            <div style="display: flex; gap: 15px; padding: 15px; background: #d1ecf1; border-left: 4px solid #17a2b8; border-radius: 4px;">
                                <div style="font-size: 24px; flex-shrink: 0;">2️⃣</div>
                                <div style="flex: 1;">
                                    <strong style="color: #0c5460; display: block; margin-bottom: 5px;">Step 2: Generate Payroll</strong>
                                    <p style="margin: 0 0 10px; color: #0c5460; font-size: 14px;">Calculate salaries for the current month based on attendance and salary structures.</p>
                                    <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=generate'); ?>" class="hrm-payroll-btn hrm-payroll-btn-primary">
                                        <span class="dashicons dashicons-calculator"></span> Generate Payroll Now
                                    </a>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid var(--hrm-border-light);">
                        <div style="background: #f0f9ff; border: 1px solid #bae6fd; border-radius: 8px; padding: 20px; margin-bottom: 15px;">
                            <h4 style="margin: 0 0 10px; color: #0369a1; font-size: 15px;">
                                <span class="dashicons dashicons-lightbulb" style="color: #0ea5e9;"></span> Want to see how it works?
                            </h4>
                            <p style="margin: 0 0 15px; color: #075985; font-size: 14px;">
                                Generate sample data to explore the payroll module with realistic demo employees and payroll records.
                            </p>
                            <button id="hrm-generate-demo-data-btn" class="hrm-payroll-btn hrm-payroll-btn-primary" style="background: linear-gradient(135deg, #0ea5e9 0%, #0284c7 100%);">
                                <span class="dashicons dashicons-admin-users"></span> Generate Demo Data
                            </button>
                            <span id="demo-data-loading" style="display: none; margin-left: 15px; color: #0369a1;">
                                <span class="dashicons dashicons-update spin"></span> Generating...
                            </span>
                        </div>
                        <p style="color: var(--hrm-text-secondary); font-size: 13px; margin: 0;">
                            <span class="dashicons dashicons-info" style="color: var(--hrm-primary);"></span>
                            Need help? Check the <a href="<?php echo admin_url('admin.php?page=hrm-payroll&tab=settings'); ?>" style="color: var(--hrm-primary);">Payroll Settings</a> to configure tax rules and working days.
                        </p>
                    </div>
                    
                    <script>
                    jQuery(document).ready(function($) {
                        $('#hrm-generate-demo-data-btn').on('click', function() {
                            if (!confirm('This will create 8 demo employees with salary structures and 3 months of payroll data. Continue?')) return;
                            
                            const btn = $(this);
                            btn.prop('disabled', true);
                            $('#demo-data-loading').show();
                            
                            $.post(ajaxurl, {
                                action: 'hrm_generate_demo_data',
                                nonce: '<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>'
                            }, function(res) {
                                if (res.success) {
                                    hrmToast('success', 'Demo data generated successfully!');
                                    setTimeout(() => location.reload(), 1500);
                                } else {
                                    hrmToast('error', res.data || 'Failed to generate demo data');
                                    btn.prop('disabled', false);
                                    $('#demo-data-loading').hide();
                                }
                            }).fail(function() {
                                hrmToast('error', 'Request failed. Please try again.');
                                btn.prop('disabled', false);
                                $('#demo-data-loading').hide();
                            });
                        });
                    });
                    </script>
                </div>
            </div>
            <?php endif; ?>

            <!-- Quick Actions Hub -->
            <div class="hrm-card-enhanced">
                <div class="hrm-card-header-enhanced" style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <h3>Payroll Command Center</h3>
                        <p style="margin: 4px 0 0; font-size: 13px; color: var(--hrm-text-secondary);">Manage your payroll operations and configurations.</p>
                    </div>
                    <a href="?page=hrm-payroll&tab=settings" class="hrm-payroll-btn hrm-payroll-btn-secondary">
                        <span class="dashicons dashicons-admin-generic"></span> Settings
                    </a>
                </div>
                <div class="hrm-card-body-enhanced">
                    <div class="hrm-payroll-quick-actions">
                        <a href="?page=hrm-payroll&tab=generate" class="hrm-payroll-action-card">
                            <div class="hrm-payroll-action-header">
                                <div class="hrm-payroll-action-icon">
                                    <span class="dashicons dashicons-calculator"></span>
                                </div>
                                <h4 class="hrm-payroll-action-title">Generate Payroll</h4>
                            </div>
                            <p class="hrm-payroll-action-desc">Run salary calculation cycles for the current period and review payouts.</p>
                        </a>

                        <a href="?page=hrm-payroll&tab=salary_structures" class="hrm-payroll-action-card">
                            <div class="hrm-payroll-action-header">
                                <div class="hrm-payroll-action-icon">
                                    <span class="dashicons dashicons-index-card"></span>
                                </div>
                                <h4 class="hrm-payroll-action-title">Salary Structures</h4>
                            </div>
                            <p class="hrm-payroll-action-desc">Define base salaries, allowances, and deductions for each employee.</p>
                        </a>

                        <a href="?page=hrm-payroll&tab=history" class="hrm-payroll-action-card">
                            <div class="hrm-payroll-action-header">
                                <div class="hrm-payroll-action-icon">
                                    <span class="dashicons dashicons-backup"></span>
                                </div>
                                <h4 class="hrm-payroll-action-title">Payroll History</h4>
                            </div>
                            <p class="hrm-payroll-action-desc">Access past payroll records, audit logs, and generated payslips.</p>
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    private static function render_salary_structures() {
        global $wpdb;
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        
        $per_page = 15;
        $current_page = HRM_Pagination::get_current_page();
        $offset = ($current_page - 1) * $per_page;

        $query = "SELECT p.ID, p.post_title FROM {$wpdb->posts} p WHERE p.post_type = 'hrm_employee' AND p.post_status = 'publish'";
        if ($search) {
            $query .= $wpdb->prepare(" AND p.post_title LIKE %s", '%' . $wpdb->esc_like($search) . '%');
        }
        $query .= " LIMIT $offset, $per_page";
        
        $employees = $wpdb->get_results($query);
        $total_employees = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");

        ?>
        <div class="hrm-filter-bar-enhanced">
            <div class="hrm-filter-group">
                <div class="hrm-search-input">
                    <input type="text" id="hrm-search-employee-structure" class="hrm-input" placeholder="Search employee..." value="<?php echo esc_attr($search); ?>">
                </div>
                <button class="hrm-payroll-btn hrm-payroll-btn-primary" onclick="location.href='?page=hrm-payroll&tab=salary_structures&s=' + jQuery('#hrm-search-employee-structure').val()">
                    <span class="dashicons dashicons-search"></span> Search
                </button>
                <?php if($search): ?>
                    <a href="?page=hrm-payroll&tab=salary_structures" class="hrm-payroll-btn hrm-payroll-btn-secondary">Clear</a>
                <?php endif; ?>
            </div>
        </div>

        <div class="hrm-table-wrapper">
            <table class="hrm-table">
                <thead>
                    <tr>
                        <th>Employee</th>
                        <th>Basic Salary</th>
                        <th>Allowances</th>
                        <th>Deductions</th>
                        <th>Last Updated</th>
                        <th style="text-align: right;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if($employees): foreach ($employees as $emp): 
                        $structure = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}hrm_salary_structures WHERE employee_id = %d", $emp->ID));
                        $allowances = $structure ? json_decode($structure->allowances, true) : [];
                        $deductions = $structure ? json_decode($structure->deductions, true) : [];
                    ?>
                    <tr>
                        <td>
                            <div style="display: flex; flex-direction: column;">
                                <span style="font-weight: 700; color: var(--hrm-text-primary);"><?php echo esc_html($emp->post_title); ?></span>
                                <span style="font-size: 11px; color: var(--hrm-text-secondary);">ID: #<?php echo $emp->ID; ?></span>
                            </div>
                        </td>
                        <td><span style="font-weight: 600; color: var(--hrm-primary);"><?php echo hrm_currency_symbol() . number_format($structure ? $structure->basic_salary : 0, 2); ?></span></td>
                        <td><span class="hrm-badge hrm-badge-info"><?php echo count((array)$allowances); ?> Items</span></td>
                        <td><span class="hrm-badge hrm-badge-warning"><?php echo count((array)$deductions); ?> Items</span></td>
                        <td><span style="font-size: 12px; color: var(--hrm-text-secondary);"><?php echo $structure ? date('M d, Y', strtotime($structure->updated_at)) : '<span class="hrm-badge hrm-badge-neutral">Not Set</span>'; ?></span></td>
                        <td style="text-align: right;">
                            <button class="hrm-payroll-btn hrm-payroll-btn-secondary btn-sm hrm-edit-structure" 
                                data-id="<?php echo $emp->ID; ?>"
                                data-name="<?php echo esc_attr($emp->post_title); ?>"
                                data-basic="<?php echo $structure ? $structure->basic_salary : 0; ?>"
                                data-overtime="<?php echo $structure ? $structure->overtime_rate : 0; ?>"
                                data-allowances='<?php echo $structure ? esc_attr($structure->allowances) : "[]"; ?>'
                                data-deductions='<?php echo $structure ? esc_attr($structure->deductions) : "[]"; ?>'>
                                <span class="dashicons dashicons-edit"></span> Manage
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="6" style="text-align:center; padding: 40px;">No employees found.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
            <?php 
            HRM_Pagination::render([
                'total_items' => $total_employees,
                'current_page' => $current_page,
                'per_page' => $per_page
            ]);
            ?>
        </div>

        <!-- Structure Modal -->
        <div id="hrm-structure-modal" class="hrm-modal" style="display:none;">
            <div class="hrm-modal-content hrm-premium-modal">
                <div class="hrm-modal-header-premium">
                    <div class="header-icon-wrapper"><span class="dashicons dashicons-index-card"></span></div>
                    <div class="header-text">
                        <h2 id="modal-emp-name">Update Salary Structure</h2>
                        <p>Define base salary and components for this employee.</p>
                    </div>
                    <button class="hrm-modal-close-premium" onclick="closeModal('hrm-structure-modal')"><span class="dashicons dashicons-no-alt"></span></button>
                </div>
                <form id="hrm-salary-structure-form" class="hrm-premium-form-modal">
                    <?php wp_nonce_field('hrm_ajax_nonce', 'nonce'); ?>
                    <input type="hidden" name="action" value="hrm_save_salary_structure">
                    <input type="hidden" name="employee_id" id="struct-emp-id">
                    
                    <div class="hrm-form-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 30px;">
                        <div class="hrm-form-group">
                            <label class="hrm-form-label">Basic Salary (Monthly) <span class="required">*</span></label>
                            <input type="number" class="hrm-input" name="basic_salary" id="struct-basic" step="0.01" required>
                        </div>
                        <div class="hrm-form-group">
                            <label class="hrm-form-label">Overtime Rate (Per Hour)</label>
                            <input type="number" class="hrm-input" name="overtime_rate" id="struct-overtime" step="0.01">
                        </div>
                    </div>

                    <div class="hrm-structure-sections-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; border-top: 1px solid var(--hrm-border-light); padding-top: 24px;">
                        <div class="hrm-struct-col">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                                <h4 style="margin: 0; font-size: 15px; color: var(--hrm-text-primary);">Allowances</h4>
                                <button type="button" class="hrm-payroll-btn hrm-payroll-btn-secondary btn-sm add-struct-item" data-type="allowance">
                                    <span class="dashicons dashicons-plus"></span>
                                </button>
                            </div>
                            <div id="allowances-list" class="struct-items-list"></div>
                        </div>
                        <div class="hrm-struct-col">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                                <h4 style="margin: 0; font-size: 15px; color: var(--hrm-text-primary);">Deductions</h4>
                                <button type="button" class="hrm-payroll-btn hrm-payroll-btn-secondary btn-sm add-struct-item" data-type="deduction">
                                    <span class="dashicons dashicons-plus"></span>
                                </button>
                            </div>
                            <div id="deductions-list" class="struct-items-list"></div>
                        </div>
                    </div>

                    <div class="hrm-modal-footer-premium" style="margin-top: 30px; border-top: 1px solid var(--hrm-border-light); padding-top: 24px; display: flex; justify-content: flex-end; gap: 12px;">
                        <button type="button" class="hrm-payroll-btn hrm-payroll-btn-secondary" onclick="closeModal('hrm-structure-modal')">Cancel</button>
                        <button type="submit" class="hrm-payroll-btn hrm-payroll-btn-primary">Save Structure</button>
                    </div>
                </form>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('.hrm-edit_structure').on('click', function() {
                // Not actually using this yet, see below for correct class
            });

            $(document).on('click', '.hrm-edit-structure', function() {
                const btn = $(this);
                $('#struct-emp-id').val(btn.data('id'));
                $('#modal-emp-name').text(btn.data('name'));
                $('#struct-basic').val(btn.data('basic'));
                $('#struct-overtime').val(btn.data('overtime'));
                
                renderStructItems('allowance', btn.data('allowances'));
                renderStructItems('deduction', btn.data('deductions'));
                
                $('#hrm-structure-modal').fadeIn().css('display', 'flex');
            });

            function renderStructItems(type, items) {
                const container = $(`#${type}s-list`);
                container.empty();
                items.forEach((item, index) => {
                    container.append(createItemHtml(type, item.label, item.value));
                });
            }

            function createItemHtml(type, label = '', value = 0) {
                return `
                <div class="struct-item-row" style="display:grid; grid-template-columns: 1fr 1fr 42px; gap:10px; margin-bottom:10px; align-items:center;">
                    <input type="text" name="${type}_labels[]" class="hrm-input" value="${label}" placeholder="Label (e.g. HRA)" required>
                    <input type="number" name="${type}_values[]" class="hrm-input" value="${value}" step="0.01" placeholder="Value" required>
                    <button type="button" class="remove-item hrm-payroll-btn hrm-payroll-btn-danger btn-sm" style="background:#fee2e2; color:#ef4444; border-color:#fecaca; height: 42px; justify-content: center;">&times;</button>
                </div>`;
            }

            $('.add-struct-item').on('click', function() {
                const type = $(this).data('type');
                $(`#${type}s-list`).append(createItemHtml(type));
            });

            $(document).on('click', '.remove-item', function() {
                $(this).closest('.struct-item-row').remove();
            });

        });
        </script>
        <style>
        .struct-items-list { margin-bottom: 10px; }
        .struct-item-row { display: flex; gap: 10px; margin-bottom: 8px; align-items: center; }
        .struct-item-row input[type="text"] { flex: 2; }
        .struct-item-row input[type="number"] { flex: 1; }
        .remove-item { background: #fee2e2; color: #ef4444; border: none; border-radius: 4px; padding: 4px 8px; cursor: pointer; }
        .hrm-structure-sections-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 24px; margin-top: 20px; border-top: 1px solid #eee; padding-top: 20px; }
        </style>
        <?php
    }

    private static function render_generate_payroll() {
        global $wpdb;
        $month = isset($_GET['month']) ? sanitize_text_field($_GET['month']) : date('Y-m');
        
        $employees = $wpdb->get_results("SELECT ID, post_title FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
        ?>
        <div class="hrm-card-enhanced" style="margin-bottom: 24px;">
            <div class="hrm-card-header-enhanced">
                <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 20px;">
                    <div>
                        <h3>Generate Payroll Rundown</h3>
                        <p style="margin: 4px 0 0; font-size: 13px; color: var(--hrm-text-secondary);">Calculate and review payouts for <strong><?php echo date('F Y', strtotime($month)); ?></strong></p>
                    </div>
                    <div style="display: flex; gap: 12px; align-items: center;">
                        <input type="month" id="payroll-month-select" value="<?php echo esc_attr($month); ?>" class="hrm-input" style="width: 160px; height: 38px;">
                        <button class="hrm-payroll-btn hrm-payroll-btn-secondary" onclick="window.location.href='?page=hrm-payroll&tab=generate&month=' + document.getElementById('payroll-month-select').value">
                            <span class="dashicons dashicons-update"></span> Change
                        </button>
                        <button class="hrm-payroll-btn hrm-payroll-btn-primary" id="bulk-generate-btn">
                            <span class="dashicons dashicons-calculator"></span> Run Bulk Cycle
                        </button>
                    </div>
                </div>
            </div>
            
            <div id="generation-progress" style="display:none; padding: 24px; border-top: 1px solid var(--hrm-border-light);">
                <div style="display:flex; justify-content:space-between; margin-bottom:12px;">
                    <span id="progress-text" style="font-weight:600; font-size:14px; color: var(--hrm-text-primary);">Processing employees...</span>
                    <span id="progress-percent" style="font-weight:700; color:var(--hrm-primary);">0%</span>
                </div>
                <div class="hrm-progress-bar-container" style="background:var(--hrm-gray-200); height:10px; border-radius:5px; overflow:hidden;">
                    <div id="progress-indicator" style="background:var(--hrm-gradient-primary); height:100%; width:0%; transition: width 0.3s cubic-bezier(0.4, 0, 0.2, 1);"></div>
                </div>
            </div>
        </div>

        <div class="hrm-table-wrapper">
             <table class="hrm-table">
                <thead>
                    <tr>
                        <th>Employee</th>
                        <th>Atten. Days</th>
                        <th>Base Salary</th>
                        <th>Allowances</th>
                        <th>Deductions</th>
                        <th>Net Payout</th>
                        <th>Status</th>
                        <th style="text-align: right;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if($employees): foreach ($employees as $emp): 
                        $payroll = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}hrm_payroll WHERE employee_id = %d AND payroll_month = %s", $emp->ID, $month));
                        
                        // Count attendance days
                        $first_day = $month . '-01';
                        $last_day = date('Y-m-t', strtotime($first_day));
                        $present_days = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date BETWEEN %s AND %s AND status IN ('present', 'late', 'leave', 'half_day')", $emp->ID, $first_day, $last_day)) ?: 0;
                    ?>
                    <tr>
                        <td><strong><?php echo esc_html($emp->post_title); ?></strong></td>
                        <td><span class="hrm-badge hrm-badge-primary"><?php echo $present_days; ?> Days</span></td>
                        <td><?php echo $payroll ? hrm_currency_symbol() . number_format($payroll->basic_salary, 2) : '-'; ?></td>
                        <td><?php echo $payroll ? '<span class="text-success">+' . hrm_currency_symbol() . number_format($payroll->total_allowances, 2) . '</span>' : '-'; ?></td>
                        <td><?php echo $payroll ? '<span class="text-danger">-' . hrm_currency_symbol() . number_format($payroll->total_deductions, 2) . '</span>' : '-'; ?></td>
                        <td><strong style="color: var(--hrm-text-primary); font-size: 15px;"><?php echo $payroll ? hrm_currency_symbol() . number_format($payroll->net_salary, 2) : '-'; ?></strong></td>
                        <td><?php echo $payroll ? HRM_UI_Helper::render_status_badge($payroll->status) : '<span class="hrm-badge hrm-badge-neutral">Pending</span>'; ?></td>
                        <td style="text-align: right;">
                            <div class="hrm-actions-flex" style="justify-content: flex-end; gap: 8px;">
                                <button class="hrm-payroll-btn hrm-payroll-btn-primary btn-sm hrm-run-single" data-id="<?php echo $emp->ID; ?>" data-month="<?php echo $month; ?>" title="Calculate Salary">
                                    <span class="dashicons dashicons-controls-play"></span> Run
                                </button>
                                <?php if ($payroll): ?>
                                    <button class="hrm-payroll-btn hrm-payroll-btn-secondary btn-sm hrm-view-payslip" data-id="<?php echo $payroll->id; ?>" title="View Preview">
                                        <span class="dashicons dashicons-visibility"></span> 
                                    </button>
                                    <button class="hrm-payroll-btn hrm-payroll-btn-danger btn-sm hrm-delete-payroll-record" style="background: #fee2e2; border-color: #fecaca; color: #ef4444;" data-id="<?php echo $payroll->id; ?>" title="Reset Record">
                                        <span class="dashicons dashicons-trash"></span>
                                    </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="8" style="text-align:center; padding: 40px;">No employees to process.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('.hrm-run-single').on('click', function() {
                const btn = $(this);
                const originalHtml = btn.html();
                btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span>');
                
                $.post(ajaxurl, {
                    action: 'hrm_generate_payroll',
                    employee_id: btn.data('id'),
                    month: btn.data('month'),
                    nonce: '<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>'
                }, function(res) {
                    if (res.success) {
                        hrmToast('success', 'Payroll calculated.');
                        location.reload();
                    } else {
                        hrmToast('error', res.data);
                        btn.prop('disabled', false).html(originalHtml);
                    }
                });
            });

            $('#bulk-generate-btn').on('click', function() {
                if (!confirm('Run payroll for all employees for ' + $('#payroll-month-select').val() + '?')) return;
                
                const btn = $(this);
                btn.prop('disabled', true);
                $('#generation-progress').show();
                
                const employees = <?php echo json_encode(array_column($employees, 'ID')); ?>;
                const month = $('#payroll-month-select').val();
                let current = 0;

                function processNext() {
                    if (current >= employees.length) {
                        $('#progress-text').text('Payroll generation complete!');
                        setTimeout(() => location.reload(), 1500);
                        return;
                    }

                    const empId = employees[current];
                    const progress = ((current + 1) / employees.length) * 100;
                    $('#progress-indicator').css('width', progress + '%');
                    $('#progress-text').text('Processing ' + (current + 1) + ' of ' + employees.length);

                    $.post(ajaxurl, {
                        action: 'hrm_generate_payroll',
                        employee_id: empId,
                        month: month,
                        nonce: '<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>'
                    }).always(() => {
                        current++;
                        processNext();
                    });
                }

                processNext();
            });

            $(document).on('click', '.hrm-delete-payroll-record', function() {
                if (!confirm('Are you sure you want to delete this payroll record?')) return;
                $.post(ajaxurl, {
                    action: 'hrm_delete_payroll',
                    id: $(this).data('id'),
                    nonce: '<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>'
                }, function(res) {
                    if (res.success) location.reload();
                });
            });
        });
        </script>
        <?php
    }

    private static function render_payroll_history() {
        global $wpdb;
        $month = isset($_GET['hmonth']) ? sanitize_text_field($_GET['hmonth']) : '';
        
        $query = "SELECT pr.*, p.post_title as employee_name FROM {$wpdb->prefix}hrm_payroll pr 
                  JOIN {$wpdb->posts} p ON pr.employee_id = p.ID 
                  WHERE 1=1";
        if ($month) {
            $query .= $wpdb->prepare(" AND pr.payroll_month = %s", $month);
        }
        $query .= " ORDER BY pr.payroll_month DESC, p.post_title ASC";
        
        $results = $wpdb->get_results($query);
        ?>
        <div class="hrm-filter-bar-enhanced">
            <div class="hrm-filter-group">
                <div style="display:flex; gap:12px; align-items:center;">
                    <label style="font-weight:700; color:var(--hrm-text-secondary); font-size: 13px; text-transform: uppercase;">Filter by Month:</label>
                    <input type="month" id="history-month" class="hrm-input" style="width: 160px;" value="<?php echo esc_attr($month); ?>">
                </div>
                <button class="hrm-payroll-btn hrm-payroll-btn-primary" onclick="location.href='?page=hrm-payroll&tab=history&hmonth=' + document.getElementById('history-month').value">
                    <span class="dashicons dashicons-filter"></span> Apply Filter
                </button>
                <?php if($month): ?>
                    <button class="hrm-payroll-btn hrm-payroll-btn-secondary" onclick="location.href='?page=hrm-payroll&tab=history'">
                        <span class="dashicons dashicons-no-alt"></span> Reset
                    </button>
                <?php endif; ?>
            </div>
        </div>

        <div class="hrm-table-wrapper">
            <table class="hrm-table">
                <thead>
                    <tr>
                        <th>Period</th>
                        <th>Employee</th>
                        <th>Net Salary</th>
                        <th>Status</th>
                        <th>Processed On</th>
                        <th style="text-align: right;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($results): foreach ($results as $row): ?>
                    <tr>
                        <td><strong><?php echo date('F Y', strtotime($row->payroll_month . '-01')); ?></strong></td>
                        <td><?php echo esc_html($row->employee_name); ?></td>
                        <td><span style="font-weight: 700; color: var(--hrm-primary);"><?php echo hrm_currency_symbol() . number_format($row->net_salary, 2); ?></span></td>
                        <td><?php echo HRM_UI_Helper::render_status_badge($row->status); ?></td>
                        <td><span style="font-size: 12px; color: var(--hrm-text-secondary);"><?php echo $row->status === 'approved' ? date('M d, Y', strtotime($row->updated_at)) : '<span class="hrm-badge hrm-badge-neutral">Draft</span>'; ?></span></td>
                        <td style="text-align: right;">
                            <div class="hrm-actions-flex" style="justify-content: flex-end; gap: 8px;">
                                <button class="hrm-payroll-btn hrm-payroll-btn-secondary btn-sm hrm-view-payslip" data-id="<?php echo $row->id; ?>" title="View & Print Statement">
                                    <span class="dashicons dashicons-visibility"></span> View Details
                                </button>
                                <?php if ($row->status === 'draft'): ?>
                                    <button class="hrm-payroll-btn hrm-payroll-btn-primary btn-sm hrm-approve-payroll" style="background: var(--hrm-gradient-success);" data-id="<?php echo $row->id; ?>">
                                        <span class="dashicons dashicons-yes-alt"></span> Approve
                                    </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; else: ?>
                    <tr>
                        <td colspan="6" style="padding: 0;">
                            <div style="text-align: center; padding: 60px 40px; background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);">
                                <div style="font-size: 64px; opacity: 0.3; margin-bottom: 20px;">📋</div>
                                <h3 style="color: var(--hrm-text-primary); margin-bottom: 10px; font-size: 20px;">No Payroll Records Found</h3>
                                <p style="color: var(--hrm-text-secondary); margin-bottom: 25px; font-size: 14px;">
                                    <?php if ($month): ?>
                                        No payroll records exist for <strong><?php echo date('F Y', strtotime($month . '-01')); ?></strong>.
                                    <?php else: ?>
                                        No payroll records have been generated yet.
                                    <?php endif; ?>
                                </p>
                                <div style="display: flex; gap: 12px; justify-content: center; flex-wrap: wrap;">
                                    <?php if ($month): ?>
                                        <a href="?page=hrm-payroll&tab=history" class="hrm-payroll-btn hrm-payroll-btn-secondary">
                                            <span class="dashicons dashicons-no-alt"></span> Clear Filter
                                        </a>
                                    <?php endif; ?>
                                    <a href="?page=hrm-payroll&tab=generate" class="hrm-payroll-btn hrm-payroll-btn-primary">
                                        <span class="dashicons dashicons-calculator"></span> Generate Payroll
                                    </a>
                                </div>
                            </div>
                        </td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php
    }

    // AJAX Handlers
    public static function handle_save_salary_structure() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        global $wpdb;
        $employee_id = intval($_POST['employee_id']);
        $basic_salary = floatval($_POST['basic_salary']);
        $overtime_rate = floatval($_POST['overtime_rate']);

        $allowances = [];
        if (!empty($_POST['allowance_labels'])) {
            foreach ($_POST['allowance_labels'] as $i => $label) {
                if ($label) $allowances[] = ['label' => sanitize_text_field($label), 'value' => floatval($_POST['allowance_values'][$i])];
            }
        }

        $deductions = [];
        if (!empty($_POST['deduction_labels'])) {
            foreach ($_POST['deduction_labels'] as $i => $label) {
                if ($label) $deductions[] = ['label' => sanitize_text_field($label), 'value' => floatval($_POST['deduction_values'][$i])];
            }
        }

        $existing = $wpdb->get_var($wpdb->prepare("SELECT id FROM {$wpdb->prefix}hrm_salary_structures WHERE employee_id = %d", $employee_id));
        
        $data = [
            'employee_id' => $employee_id,
            'basic_salary' => $basic_salary,
            'overtime_rate' => $overtime_rate,
            'allowances' => json_encode($allowances),
            'deductions' => json_encode($deductions),
            'updated_at' => current_time('mysql')
        ];

        if ($existing) {
            $wpdb->update("{$wpdb->prefix}hrm_salary_structures", $data, ['id' => $existing]);
        } else {
            $wpdb->insert("{$wpdb->prefix}hrm_salary_structures", $data);
        }

        wp_send_json_success('Salary structure updated.');
    }

    public static function handle_generate_payroll() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        global $wpdb;
        $employee_id = intval($_POST['employee_id']);
        $month = sanitize_text_field($_POST['month']); // YYYY-MM
        
        $settings = get_option('hrm_payroll_settings', [
            'working_days' => 22,
            'tax_slabs' => [],
            'currency' => '$'
        ]);

        // 1. Get Salary Structure
        $structure = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}hrm_salary_structures WHERE employee_id = %d", $employee_id));
        if (!$structure) wp_send_json_error('Salary structure not set for this employee.');

        // 2. Get Attendance Data
        $first_day = $month . '-01';
        $last_day = date('Y-m-t', strtotime($first_day));
        $total_working_days = $settings['working_days'];
        
        $present_days = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date BETWEEN %s AND %s AND status IN ('present', 'late', 'leave')", $employee_id, $first_day, $last_day)) ?: 0;
        $half_days = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date BETWEEN %s AND %s AND status = 'half_day'", $employee_id, $first_day, $last_day)) ?: 0;
        
        $effective_days = $present_days + ($half_days * 0.5);
        
        // 3. Calculation
        $basic = (float)$structure->basic_salary;
        $day_rate = $basic / $total_working_days;
        
        // Penalty for missing days
        $absents = max(0, $total_working_days - $effective_days);
        $penalty = $absents * $day_rate;

        $allowances_arr = json_decode($structure->allowances, true) ?: [];
        $total_allowance = 0;
        foreach($allowances_arr as $a) $total_allowance += (float)$a['value'];

        $deductions_arr = json_decode($structure->deductions, true) ?: [];
        $total_deduction = 0;
        foreach($deductions_arr as $d) $total_deduction += (float)$d['value'];

        // Overtime
        $overtime_pay = 0;
        if ($structure->overtime_rate > 0) {
             $total_hours = $wpdb->get_var($wpdb->prepare("SELECT SUM(hours_worked) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date BETWEEN %s AND %s", $employee_id, $first_day, $last_day)) ?: 0;
             $expected_hours = $effective_days * 8; // Assuming 8h shift
             $ot_hours = max(0, $total_hours - $expected_hours);
             $overtime_pay = $ot_hours * (float)$structure->overtime_rate;
        }

        // Gross Income
        $gross_monthly = $basic + $total_allowance + $overtime_pay - $penalty;
        $annual_taxable = ($gross_monthly - $total_deduction) * 12;

        // Tax Calculation based on slabs
        $annual_tax = 0;
        if (!empty($settings['tax_slabs'])) {
            foreach ($settings['tax_slabs'] as $slab) {
                if ($annual_taxable > $slab['min']) {
                    $taxable_in_slab = min($annual_taxable, $slab['max'] ?: INF) - $slab['min'];
                    $annual_tax += ($taxable_in_slab * $slab['percent'] / 100);
                }
            }
        }
        $monthly_tax = $annual_tax / 12;

        // Net Salary
        $net = $gross_monthly - $total_deduction - $monthly_tax;
        $net = max(0, $net);

        // Save Snapshot
        $payslip_data = [
            'meta' => [
                'employee_name' => get_the_title($employee_id),
                'month' => $month,
                'working_days' => $total_working_days,
                'present_days' => $present_days,
                'half_days' => $half_days,
                'ot_hours' => isset($ot_hours) ? round($ot_hours, 1) : 0
            ],
            'earnings' => [
                ['label' => 'Basic Salary', 'value' => $basic],
                ['label' => 'Allowances', 'items' => $allowances_arr, 'total' => $total_allowance],
                ['label' => 'Overtime Pay', 'value' => $overtime_pay]
            ],
            'deductions' => [
                ['label' => 'Attendance Penalty', 'value' => $penalty],
                ['label' => 'Tax (Monthly)', 'value' => $monthly_tax],
                ['label' => 'Other Deductions', 'items' => $deductions_arr, 'total' => $total_deduction]
            ],
            'net_salary' => $net
        ];

        $existing = $wpdb->get_var($wpdb->prepare("SELECT id FROM {$wpdb->prefix}hrm_payroll WHERE employee_id = %d AND payroll_month = %s", $employee_id, $month));
        
        $data = [
            'employee_id' => $employee_id,
            'payroll_month' => $month,
            'basic_salary' => $basic,
            'total_allowances' => $total_allowance,
            'total_deductions' => $total_deduction,
            'attendance_penalty' => $penalty,
            'overtime_pay' => $overtime_pay,
            'net_salary' => $net,
            'payslip_data' => json_encode($payslip_data),
            'status' => 'draft',
            'created_by' => get_current_user_id()
        ];

        if ($existing) {
            $wpdb->update("{$wpdb->prefix}hrm_payroll", $data, ['id' => $existing]);
        } else {
            $wpdb->insert("{$wpdb->prefix}hrm_payroll", $data);
        }

        wp_send_json_success('Payroll generated for employee.');
    }

    public static function handle_update_payroll_status() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        global $wpdb;
        $id = intval($_POST['id']);
        $status = sanitize_text_field($_POST['status']);

        $wpdb->update("{$wpdb->prefix}hrm_payroll", ['status' => $status, 'updated_at' => current_time('mysql')], ['id' => $id]);
        wp_send_json_success('Status updated.');
    }

    public static function handle_delete_payroll() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        global $wpdb;
        $id = intval($_POST['id']);
        $wpdb->delete("{$wpdb->prefix}hrm_payroll", ['id' => $id]);
        wp_send_json_success('Record deleted.');
    }

    public static function handle_get_payslip() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        
        global $wpdb;
        $id = intval($_POST['id']);
        $row = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}hrm_payroll WHERE id = %d", $id));
        
        if (!$row) wp_send_json_error('Not found');

        // RBAC check
        if (!current_user_can('manage_options')) {
            $employee_id = self::get_current_employee_id();
            if (!$employee_id || $employee_id != $row->employee_id) {
                 wp_send_json_error('Permission denied');
            }
        }

        $data = json_decode($row->payslip_data, true);
        
        ob_start();
        ?>
        <div class="payslip-inner">
            <div class="payslip-header">
                <div class="payslip-title">
                     <h3><?php echo strtoupper(get_bloginfo('name')); ?></h3>
                     <p style="margin:5px 0 0; color:var(--text-secondary); font-weight:500;">PAYROLL ADVICE - <?php echo date('F Y', strtotime($row->payroll_month . '-01')); ?></p>
                </div>
                <div style="text-align:right;">
                    <div style="font-size: 18px; font-weight: 700; color: var(--text-main);">#PAY-<?php echo str_pad($row->id, 6, '0', STR_PAD_LEFT); ?></div>
                    <div style="font-size: 13px; color: var(--text-secondary); margin-top: 5px;">Payment Date: <?php echo date('Y-m-d', strtotime($row->updated_at)); ?></div>
                    <div class="no-print" style="margin-top:10px;">
                        <?php echo HRM_UI_Helper::render_status_badge($row->status); ?>
                    </div>
                </div>
            </div>

            <div class="payslip-employee-info" style="margin-bottom: 40px; display:flex; justify-content:space-between; background:#f8fafc; padding:25px; border-radius:12px; border: 1px solid #e2e8f0;">
                <div style="flex:1;">
                    <h5 style="margin:0 0 10px; color:var(--text-secondary); text-transform:uppercase; font-size:11px; letter-spacing:1px;">Employee Details</h5>
                    <div style="font-size:16px; font-weight:700; color:var(--text-main); margin-bottom:5px;"><?php echo esc_html($data['meta']['employee_name']); ?></div>
                    <div style="color:var(--text-secondary); font-size:14px;"><?php echo get_post_meta($row->employee_id, 'hrm_position', true) ?: 'Staff Member'; ?></div>
                    <div style="color:var(--text-secondary); font-size:13px; margin-top:5px;">ID: <?php echo $row->employee_id; ?></div>
                </div>
                <div style="flex:1; display:flex; flex-direction:column; align-items:flex-end;">
                    <h5 style="margin:0 0 10px; color:var(--text-secondary); text-transform:uppercase; font-size:11px; letter-spacing:1px;">Attendance Summary</h5>
                    <div style="display:grid; grid-template-columns: 1fr auto; gap:10px 20px; text-align:right; width:fit-content;">
                        <span style="color:var(--text-secondary);">Standard Days:</span> <strong style="color:var(--text-main);"><?php echo $data['meta']['working_days']; ?></strong>
                        <span style="color:var(--text-secondary);">Present:</span> <strong style="color:var(--text-main);"><?php echo $data['meta']['present_days']; ?></strong>
                        <span style="color:var(--text-secondary);">Half Days:</span> <strong style="color:var(--text-main);"><?php echo $data['meta']['half_days']; ?></strong>
                        <?php if(!empty($data['meta']['ot_hours'])): ?>
                            <span style="color:var(--text-secondary);">Overtime:</span> <strong style="color:var(--text-main);"><?php echo $data['meta']['ot_hours']; ?> hrs</strong>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="payslip-body">
                <div class="payslip-col">
                    <h4>Earnings</h4>
                    <table class="payslip-table">
                        <?php foreach($data['earnings'] as $e): ?>
                            <?php if (isset($e['items'])): ?>
                                <?php foreach($e['items'] as $item): ?>
                                    <tr>
                                        <td><?php echo esc_html($item['label']); ?></td>
                                        <td style="text-align:right; font-family: 'Courier New', Courier, monospace; font-weight:600;">
                                            <?php echo hrm_currency_symbol() . number_format($item['value'], 2); ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td><?php echo esc_html($e['label']); ?></td>
                                    <td style="text-align:right; font-family: 'Courier New', Courier, monospace; font-weight:600;">
                                        <?php echo hrm_currency_symbol() . number_format($e['value'], 2); ?>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        <?php endforeach; ?>
                        <tr class="payslip-total-row">
                            <td>Gross Earnings</td>
                            <td style="text-align:right; color:var(--primary-color);">
                                <?php echo hrm_currency_symbol() . number_format($row->basic_salary + $row->total_allowances + $row->overtime_pay, 2); ?>
                            </td>
                        </tr>
                    </table>
                </div>
                <div class="payslip-col">
                    <h4>Deductions</h4>
                    <table class="payslip-table">
                        <?php foreach($data['deductions'] as $d): ?>
                            <?php if (isset($d['items'])): ?>
                                <?php foreach($d['items'] as $item): ?>
                                    <tr>
                                        <td><?php echo esc_html($item['label']); ?></td>
                                        <td style="text-align:right; font-family: 'Courier New', Courier, monospace; color:#dc2626;">
                                            -<?php echo hrm_currency_symbol() . number_format($item['value'], 2); ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <?php if($d['value'] > 0): ?>
                                    <tr>
                                        <td><?php echo esc_html($d['label']); ?></td>
                                        <td style="text-align:right; font-family: 'Courier New', Courier, monospace; color:#dc2626;">
                                            -<?php echo hrm_currency_symbol() . number_format($d['value'], 2); ?>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            <?php endif; ?>
                        <?php endforeach; ?>
                        <tr class="payslip-total-row">
                            <td>Total Deductions</td>
                            <td style="text-align:right; color:#dc2626;">
                                <?php echo hrm_currency_symbol() . number_format($row->total_deductions + $row->attendance_penalty, 2); ?>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>

            <div class="net-salary-highlight">
                <span>NET TAKE-HOME SALARY</span>
                <strong><?php echo hrm_currency_symbol() . number_format($row->net_salary, 2); ?></strong>
            </div>

            <div style="margin-top: 40px; border-top: 1px dashed #e2e8f0; padding-top: 20px;">
                <div style="display:flex; justify-content:space-between; align-items:flex-end;">
                    <div style="font-size:12px; color:var(--text-secondary); max-width:400px;">
                        Note: This is an electronically generated document. No physical signature is required. For any discrepancies, please contact the HR department within 3 business days.
                    </div>
                    <?php if ($row->status === 'approved'): ?>
                        <div style="padding: 10px; border: 2px solid #10b981; color: #10b981; font-weight: 800; border-radius: 4px; transform: rotate(-10deg); text-transform: uppercase;">
                            FINALIZED & PAID
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
        $html = ob_get_clean();
        wp_send_json_success($html);
    }

    public static function render_employee_payroll($employee_id) {
        global $wpdb;
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}hrm_payroll WHERE employee_id = %d AND status = 'approved' ORDER BY payroll_month DESC",
            $employee_id
        ));

        ?>
        <div class="hrm-card">
            <div class="hrm-card-header">
                <h3><span class="dashicons dashicons-money-alt"></span> My Payroll History</h3>
                <p>View and download your monthly payslips.</p>
            </div>
            <div class="hrm-card-body" style="padding: 0;">
                <table class="hrm-table">
                    <thead>
                        <tr>
                            <th>Month</th>
                            <th>Net Salary</th>
                            <th>Status</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($results): foreach ($results as $row): ?>
                        <tr>
                            <td><strong><?php echo date('F Y', strtotime($row->payroll_month . '-01')); ?></strong></td>
                            <td><?php echo hrm_currency_symbol() . number_format($row->net_salary, 2); ?></td>
                            <td><?php echo HRM_UI_Helper::render_status_badge($row->status); ?></td>
                            <td>
                                <div class="hrm-actions-flex">
                                    <button class="hrm-payroll-btn hrm-payroll-btn-primary hrm-view-payslip" data-id="<?php echo $row->id; ?>">
                                        <span class="dashicons dashicons-visibility"></span> View & Download
                                    </button>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; else: ?>
                        <tr><td colspan="4" style="text-align:center; padding: 40px;">No finalized payroll records available yet.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <div id="hrm-payslip-modal" class="hrm-modal" style="display:none;">
            <div class="hrm-modal-content hrm-premium-modal hrm-payslip-container" style="max-width: 900px;">
                <div class="hrm-modal-header-premium no-print">
                    <div class="header-icon-wrapper"><span class="dashicons dashicons-money-alt"></span></div>
                    <div class="header-text">
                        <h2>My Payslip</h2>
                        <p>Earnings & Deductions Statement</p>
                    </div>
                    <div class="header-actions" style="margin-left: auto; display: flex; gap: 10px;">
                        <button class="hrm-btn-outline btn-sm" onclick="window.print()"><span class="dashicons dashicons-printer"></span> Download / Print</button>
                        <button class="hrm-modal-close-premium" onclick="jQuery('#hrm-payslip-modal').fadeOut()"><span class="dashicons dashicons-no-alt"></span></button>
                    </div>
                </div>
                <div id="payslip-content" class="printable-area">
                    <!-- Loaded via AJAX -->
                </div>
            </div>
        </div>
        <?php
    }

    private static function get_current_employee_id() {
        $current_user = wp_get_current_user();
        $employee_post = get_posts([
            'post_type' => 'hrm_employee',
            'author'    => $current_user->ID,
            'numberposts' => 1,
            'fields'    => 'ids'
        ]);
        return (!empty($employee_post)) ? $employee_post[0] : 0;
    }
    
    /**
     * Handle demo data generation
     */
    public static function handle_generate_demo_data() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized access');
        }
        
        // Include demo data class
        require_once HRM_PLUGIN_PATH . 'includes/hrm-payroll-demo-data.php';
        
        try {
            HRM_Payroll_Demo_Data::generate_demo_data();
            wp_send_json_success('Demo data generated successfully. 8 employees with 3 months of payroll data created.');
        } catch (Exception $e) {
            wp_send_json_error('Failed to generate demo data: ' . $e->getMessage());
        }
    }
    
    /**
     * Handle demo data clearing
     */
    public static function handle_clear_demo_data() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized access');
        }
        
        // Include demo data class
        require_once HRM_PLUGIN_PATH . 'includes/hrm-payroll-demo-data.php';
        
        try {
            HRM_Payroll_Demo_Data::clear_demo_data();
            wp_send_json_success('All demo data cleared successfully.');
        } catch (Exception $e) {
            wp_send_json_error('Failed to clear demo data: ' . $e->getMessage());
        }
    }
}

// Utility function if not exists
if (!function_exists('hrm_currency_symbol')) {
    function hrm_currency_symbol() {
        $settings = get_option('hrm_payroll_settings', ['currency' => '$']);
        return $settings['currency'];
    }
}
